/*= Gossen Metrawatt MSP Power Supply =======================================*/
/* LabWindows/CVI 5.5 VXI PnP Instrument Driver                              */
/* Original Release: April 2000		                                         */
/* By: Marek Malohlava                                              	     */
/*     PH. +420 69 6996146               		Fax +420 69 6996147          */
/*     Email marek.malohlava@elcom.cz                                        */
/*																			 */
/* Modification History: None                                                */
/*																			 */
/*===========================================================================*/
#include <visa.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <utility.h>
#include "gmmsp.h"

#define IEEE_ERROR_BITS     	0x2C

#define GMMSP_REVISION     		"Rev 1.0, 05/2000, CVI 5.5" /* Instrument driver revision */
#define BUFFER_SIZE         	512L         /* File I/O buffer size */
#define GMMSP_DEFAULT_TIMEOUT	10000	/* Default Timeout value in ms (10 sec) */
#define DEFAULT_IO_DELAY  		0.002 
    
/*= Macros ==================================================================*/
#define CHECKERR(fCal) if (gmmsp_status = (fCal), gmmsp_status < VI_SUCCESS) \
							return gmmsp_status; else
#define CHECKERR_CU(fCal) if (gmmsp_status = (fCal), gmmsp_status < VI_SUCCESS) \
							return gmmsp_initCleanUp (rmSession, instrSession, gmmsp_status); else
#define NUMBER_OF_CHANNELS                  (8)
					
/*****************************************************************************/
/*= INSTRUMENT-DEPENDENT COMMAND ARRAYS =====================================*/
/*****************************************************************************/

 static ViString StateArr[] = { "OFF", "ON"};
 static ViString OutputModeArr[] = { "CV", "CC"}; 
 static ViString PowerModeArr[] = { "RST", "RCL", "SBY", VI_NULL}; 
 static ViString SetRegArr[] = { "*ESE", "ERAE", "ERBE", "ERCE", "ERDE" "*SRE", "*PRE"};
 static ViString GetRegArr[] = { "*ESR?", "*ESE?", "CRA?", "ERA?", "ERAE?", "CRB?", "ERB?",
 								 "ERBE?", "ERC?", "ERCE?", "ERD?", "ERDE?", "CRC?", 
 								 "*STB?", "*SRE?", "*PRE?"};
 
/*****************************************************************************/
/*= INSTRUMENT-DEPENDENT STATUS/RANGE STRUCTURE  ============================*/
/*****************************************************************************/
/* gmmsp_stringValPair is used in the gmmsp_error_message function           */
/* gmmsp_statusDataRanges is used to track session dependent status & ranges */
/*===========================================================================*/
typedef struct  gmmsp_stringValPair
{
   ViStatus stringVal;
   ViString stringName;
}  gmmsp_tStringValPair;
 
struct gmmsp_statusDataRanges 
{
    ViBoolean 	errorChecking;
    ViChar    	instrDriverRevision[256];
    ViReal64	currentLimits[8];
    ViReal64	voltageLimits[8];
    ViReal64 ioDelay;
    
};

typedef struct gmmsp_statusDataRanges *gmmsp_instrRange;

/*****************************************************************************/
/*= UTILITY ROUTINE DECLARATIONS (Non-Exportable Functions) =================*/
/*****************************************************************************/
ViBoolean gmmsp_invalidViBooleanRange (ViBoolean val);
ViBoolean gmmsp_invalidViInt32Range (ViInt32 val, ViInt32 min, ViInt32 max);
ViBoolean gmmsp_invalidViReal64Range (ViReal64 val, ViReal64 min,ViReal64 max);
ViBoolean gmmsp_invalidViString (ViString val, ViUInt32 maxLength);

ViStatus gmmsp_initCleanUp (ViSession openRMSession,
                             ViPSession openInstrSession,
	                         ViStatus currentStatus);
ViStatus gmmsp_defaultInstrSetup (ViSession openInstrSession);

/*****************************************************************************/
/*------ INSERT INSTRUMENT-DEPENDENT UTILITY ROUTINE DECLARATIONS HERE ------*/
/*****************************************************************************/
ViStatus gmmsp_checkStatus (ViSession instrSession);
ViStatus gmmsp_readChannelValue (ViSession instrSession, ViString command, 
								 ViInt32 channel, ViPReal64 value);
ViStatus gmmsp_readChannelState (ViSession instrSession, ViString command, 
								 ViInt32 channel, ViPBoolean state);	
ViStatus gmmsp_writeGroup (ViSession instrSession, ViInt32 group, ViString command);
ViStatus gmmsp_readGroup (ViSession instrSession, ViPInt32 group, ViString command);

/*****************************************************************************/
/*====== USER-CALLABLE FUNCTIONS (Exportable Functions) =====================*/
/*****************************************************************************/
/*===========================================================================*/
/* Function: Initialize                                                      */
/* Purpose:  This function opens the instrument, queries the instrument      */
/*           for its ID, and initializes the instrument to a known state.    */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_init (ViRsrc resourceName, 
							  ViBoolean IDQuery,
                    		  ViBoolean resetDevice, 
                    		  ViPSession instrSession)
{
    ViStatus	gmmsp_status = VI_SUCCESS;
    ViUInt16    interface = VI_INTF_GPIB;
    ViSession	rmSession = 0;
    ViChar		rdBuffer[BUFFER_SIZE];
    gmmsp_instrRange instrPtr;
    

    /*- Check input parameter ranges ----------------------------------------*/
    if (gmmsp_invalidViBooleanRange (IDQuery))
        return VI_ERROR_PARAMETER2;
    if (gmmsp_invalidViBooleanRange (resetDevice))
        return VI_ERROR_PARAMETER3;

    /*- Open instrument session ---------------------------------------------*/
    CHECKERR( viOpenDefaultRM (&rmSession));

    if ((gmmsp_status = viOpen (rmSession, resourceName, VI_LOAD_CONFIG,
                                 VI_NULL, instrSession)) < 0)
    	{
        viClose (rmSession);
        return gmmsp_status;
    	}

    instrPtr = malloc (sizeof (struct gmmsp_statusDataRanges));
    instrPtr->ioDelay = DEFAULT_IO_DELAY; // 2 ms
    CHECKERR (viSetAttribute (*instrSession, VI_ATTR_USER_DATA, (ViUInt32)instrPtr));
    
    /* Serial interface support */
    CHECKERR_CU( viGetAttribute (*instrSession, VI_ATTR_INTF_TYPE, &interface));

    if (interface == VI_INTF_ASRL)  /* Serial Port */
        {
        CHECKERR_CU( viSetAttribute (*instrSession, VI_ATTR_TERMCHAR, '\n'));
        CHECKERR_CU( viSetAttribute (*instrSession, VI_ATTR_TERMCHAR_EN,
                                     VI_TRUE));
        CHECKERR_CU( viSetAttribute (*instrSession, VI_ATTR_ASRL_END_IN,
                                     VI_ASRL_END_TERMCHAR));
        CHECKERR_CU( viSetAttribute (*instrSession, VI_ATTR_ASRL_END_OUT,
                                     VI_ASRL_END_TERMCHAR));
		CHECKERR_CU( viFlush (*instrSession, VI_ASRL_IN_BUF));
        }
	
    /*- Configure VISA Formatted I/O ----------------------------------------*/
    CHECKERR_CU( viSetAttribute (*instrSession, VI_ATTR_TMO_VALUE,
    							 GMMSP_DEFAULT_TIMEOUT));
    CHECKERR_CU( viSetBuf (*instrSession, VI_READ_BUF|VI_WRITE_BUF, 4000));
    CHECKERR_CU( viSetAttribute (*instrSession, VI_ATTR_WR_BUF_OPER_MODE,
                                 VI_FLUSH_ON_ACCESS));
    CHECKERR_CU( viSetAttribute (*instrSession, VI_ATTR_RD_BUF_OPER_MODE,
                                 VI_FLUSH_ON_ACCESS));
                                 
    /*- Identification Query ------------------------------------------------*/
    if (IDQuery)
    	{
        CHECKERR_CU( viPrintf (*instrSession, "*IDN?"));
        Delay(instrPtr->ioDelay);
        CHECKERR_CU( viScanf (*instrSession, "%[^\r\n]", rdBuffer));
		if (!strstr (rdBuffer, "GOSSEN-METRAWATT,64D42P"))
        	return gmmsp_initCleanUp (rmSession, instrSession,
        	                           VI_ERROR_FAIL_ID_QUERY);
    	}    	

    /*- Reset instrument ----------------------------------------------------*/
    if (resetDevice)
        gmmsp_status = gmmsp_reset (*instrSession);
    else  /*- Send Default Instrument Setup ---------------------------------*/
        gmmsp_status = gmmsp_defaultInstrSetup (*instrSession);
    CHECKERR_CU( gmmsp_status);

    return gmmsp_status;
}

/*===========================================================================*/
/* Function: Aplication Example                                              */
/* Purpose:  This function sets the voltage/current setpoints and limits and */
/*           sets the output state to On.									 */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_AppExample (ViSession instrSession, ViInt32 channel, 
									ViReal64 voltageLimit, ViReal64 currentLimit,
									ViReal64 voltage, ViReal64 current, 
									ViPReal64 nominalVoltage, ViPReal64 nominalCurrent)
{
	ViStatus gmmsp_status = VI_SUCCESS; 
	gmmsp_instrRange instrPtr;    
	ViBoolean valid;
    
    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));

    if (gmmsp_invalidViInt32Range (channel, 1, 8))
    	return VI_ERROR_PARAMETER2;
    
    if (gmmsp_invalidViReal64Range (voltageLimit, 0.0, instrPtr->voltageLimits[channel-1]))	 
       return VI_ERROR_PARAMETER3;
       
    if (gmmsp_invalidViReal64Range (currentLimit, 0.0, instrPtr->currentLimits[channel-1]))	 
       return VI_ERROR_PARAMETER4;
       
    if (gmmsp_invalidViReal64Range (voltage, 0.0, instrPtr->voltageLimits[channel-1]))	 
       	return VI_ERROR_PARAMETER5;

    if (gmmsp_invalidViReal64Range (current, 0.0, instrPtr->currentLimits[channel-1]))	 
       return VI_ERROR_PARAMETER6;

    if (!nominalVoltage)	 
       return VI_ERROR_PARAMETER7;
    
    if (!nominalCurrent)	 
       return VI_ERROR_PARAMETER8;
   	
   	CHECKERR (gmmsp_confSetCurrLimit (instrSession, channel, currentLimit));
	CHECKERR (gmmsp_confSetVoltLimit (instrSession, channel, voltageLimit));
 	CHECKERR (gmmsp_confSetCurrent (instrSession, channel, current)); 	
    CHECKERR (gmmsp_confSetVolt (instrSession, channel, voltage));
	CHECKERR (gmmsp_confSetOutputState (instrSession, channel, VI_TRUE)); 
    CHECKERR (gmmsp_measOutCurr (instrSession, channel, nominalCurrent, VI_FALSE, &valid));
    CHECKERR (gmmsp_measOutVolt (instrSession, channel, nominalVoltage, VI_FALSE, &valid));
    CHECKERR (gmmsp_checkStatus (instrSession));
    
	return gmmsp_status;
}

/*===========================================================================*/
/* Function: Set Voltage Setting                                             */
/* Purpose:  This function sets the setpoint value of the output voltage.    */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_confSetVolt (ViSession instrSession,
									 ViInt32 channel,
                                     ViReal64 voltage)
{                               
    ViStatus gmmsp_status = VI_SUCCESS;
	gmmsp_instrRange instrPtr;    

    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));

    if (gmmsp_invalidViInt32Range (channel, 1, 8))
    	return VI_ERROR_PARAMETER2;
    if (gmmsp_invalidViReal64Range (voltage, 0.0, instrPtr->voltageLimits[channel-1]))	 
       	return VI_ERROR_PARAMETER3;

    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr))
  	Delay (instrPtr->ioDelay);
	CHECKERR (viPrintf (instrSession, "US %ld,%.3f", channel, voltage));
    CHECKERR (gmmsp_checkStatus (instrSession));
    
	return gmmsp_status;
}

/*===========================================================================*/
/* Function: Get Voltage Setting                                             */
/* Purpose:  This function queries the setpoint value of the output voltage. */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_confGetVolt (ViSession instrSession,
									 ViInt32 channel,		
                                     ViPReal64 voltage)
{                               
    ViStatus gmmsp_status = VI_SUCCESS;
    
	if (gmmsp_invalidViInt32Range (channel, 1, 8))
    	return VI_ERROR_PARAMETER2;
    if (!voltage)	 
       return VI_ERROR_PARAMETER3;

   	CHECKERR (gmmsp_readChannelValue (instrSession, "US?", channel, voltage));
    CHECKERR (gmmsp_checkStatus (instrSession));
    
	return gmmsp_status;
}

/*===========================================================================*/
/* Function: Set Voltage Limit                                               */
/* Purpose:  This function defines the upper soft limit for the voltage		 */
/*			 setpoint.								                         */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_confSetVoltLimit (ViSession instrSession,
										  ViInt32 channel,	
                                          ViReal64 voltageLimit)
{                               
    ViStatus	gmmsp_status = VI_SUCCESS;
	gmmsp_instrRange	instrPtr;    

	CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
	if (gmmsp_invalidViInt32Range (channel, 1, 8))
    	return VI_ERROR_PARAMETER2;
    if (gmmsp_invalidViReal64Range (voltageLimit, 0.0, instrPtr->voltageLimits[channel-1]))	 
       return VI_ERROR_PARAMETER3;					  

	Delay (instrPtr->ioDelay);
	CHECKERR (viPrintf (instrSession, "UL %ld,%.3f", channel, voltageLimit));
    CHECKERR (gmmsp_checkStatus (instrSession));
    
	return gmmsp_status;
}

/*===========================================================================*/
/* Function: Get Voltage Limit                                               */
/* Purpose:  This function queries the upper soft limit for the voltage		 */
/*           setpoint. 													     */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_confGetVoltLimit (ViSession instrSession,
										  ViInt32 channel,
                                          ViPReal64 voltageLimit)
{                                         
    ViStatus	gmmsp_status = VI_SUCCESS;

    if (gmmsp_invalidViInt32Range (channel, 1, 8))
    	return VI_ERROR_PARAMETER2;
    if (!voltageLimit)	 
       return VI_ERROR_PARAMETER3;

   	CHECKERR (gmmsp_readChannelValue (instrSession, "UL?", channel, voltageLimit));
    CHECKERR (gmmsp_checkStatus (instrSession));

	return gmmsp_status;
}                                       

/*===========================================================================*/
/* Function: Set Current Setting                                             */
/* Purpose:  This function defines the setpoint value of the output current. */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_confSetCurrent (ViSession instrSession,
										ViInt32 channel,
                                        ViReal64 current)
{                               
    ViStatus	gmmsp_status = VI_SUCCESS;
	gmmsp_instrRange	instrPtr;    
    
	CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
	if (gmmsp_invalidViInt32Range (channel, 1, 8))
    	return VI_ERROR_PARAMETER2;
    if (gmmsp_invalidViReal64Range (current, 0.0, instrPtr->currentLimits[channel-1]))	 
       return VI_ERROR_PARAMETER3;

  	Delay (instrPtr->ioDelay);
    CHECKERR (viPrintf (instrSession, "IS %ld,%.3f", channel, current));
    CHECKERR (gmmsp_checkStatus (instrSession));
	
	return gmmsp_status;
}

/*===========================================================================*/
/* Function: Get Current Setting                                             */
/* Purpose:  This function queries the setpoint value of the output current. */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_confGetCurrent (ViSession instrSession,
										ViInt32 channel,
                                        ViPReal64 current)
{                               
    ViStatus	gmmsp_status = VI_SUCCESS;
    
    if (gmmsp_invalidViInt32Range (channel, 1, 8))
    	return VI_ERROR_PARAMETER2;
    if (!current)	 
       return VI_ERROR_PARAMETER3;

   	CHECKERR (gmmsp_readChannelValue (instrSession, "IS?", channel, current));
    CHECKERR (gmmsp_checkStatus (instrSession));
    
	return gmmsp_status;
}

/*===========================================================================*/
/* Function: Set Current Limit                                               */
/* Purpose:  This function defines the upper soft limit for the current		 */
/*			 setpoint.								     					 */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_confSetCurrLimit (ViSession instrSession,
										  ViInt32 channel,
                                          ViReal64 currentLimit)
{                               
    ViStatus	gmmsp_status = VI_SUCCESS;
	gmmsp_instrRange	instrPtr;    
    
    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
    if (gmmsp_invalidViInt32Range (channel, 1, 8))
    	return VI_ERROR_PARAMETER2;
    if (gmmsp_invalidViReal64Range (currentLimit, 0.0, instrPtr->currentLimits[channel-1]))	 
       return VI_ERROR_PARAMETER3;

  	Delay (instrPtr->ioDelay);
	CHECKERR (viPrintf (instrSession, "IL %ld,%.3f", channel, currentLimit));
    CHECKERR (gmmsp_checkStatus (instrSession));
    
	return gmmsp_status;
}

/*===========================================================================*/
/* Function: Get Current Limit                                               */
/* Purpose:  This function queries the upper soft limit for the current      */
/*			 setpoint.							 							 */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_confGetCurrLimit (ViSession instrSession,
										  ViInt32 channel,
                                          ViPReal64 currentLimit)
{                                         
    ViStatus	gmmsp_status = VI_SUCCESS;

    if (gmmsp_invalidViInt32Range (channel, 1, 8))
    	return VI_ERROR_PARAMETER2;
    if (!currentLimit)	 
       return VI_ERROR_PARAMETER3;

   	CHECKERR (gmmsp_readChannelValue (instrSession, "IL?", channel, currentLimit));
    CHECKERR (gmmsp_checkStatus (instrSession));

	return gmmsp_status;
}         

/*===========================================================================*/
/* Function: Output Complete Setting                                         */
/* Purpose:  This function is usefull for complex setting of the output.	 */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_confOutpComplSet (ViSession instrSession,
                                          ViInt32 channel, ViBoolean outputState,
                                          ViInt32 foldbackModeSetting,
                                          ViReal64 foldbackDelay,
                                          ViBoolean foldbackState,
                                          ViBoolean sinkState)
{                                         
    ViStatus	gmmsp_status = VI_SUCCESS;

    if (gmmsp_invalidViInt32Range (channel, 1, 8))
    	return VI_ERROR_PARAMETER2;
    if (gmmsp_invalidViBooleanRange (outputState))
    	return VI_ERROR_PARAMETER3;
    if (gmmsp_invalidViInt32Range (foldbackModeSetting, 0, 1))
    	return VI_ERROR_PARAMETER4;
    if (gmmsp_invalidViReal64Range (foldbackDelay, 0.0, 9.999))
    	return VI_ERROR_PARAMETER5;
    if (gmmsp_invalidViBooleanRange (foldbackState))
    	return VI_ERROR_PARAMETER6;
    if (gmmsp_invalidViBooleanRange (sinkState))
    	return VI_ERROR_PARAMETER7;
  
	CHECKERR (gmmsp_confSetOutputState (instrSession, channel, outputState));
    CHECKERR (gmmsp_confSetOutputMode (instrSession, channel, foldbackModeSetting));
    CHECKERR (gmmsp_confSetOCPDelay (instrSession, channel, foldbackDelay));
    CHECKERR (gmmsp_confSetOCPState (instrSession, channel, foldbackState));
    CHECKERR (gmmsp_confSetSinkState (instrSession, channel, sinkState));
    
    CHECKERR (gmmsp_checkStatus (instrSession));

	return gmmsp_status;
}         

/*===========================================================================*/
/* Function: Set Output State                                                */
/* Purpose:  This function selects power output ON/OFF.  					 */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_confSetOutputState (ViSession instrSession,
										    ViInt32 channel,
                                            ViBoolean state)
{                               
    ViStatus	gmmsp_status = VI_SUCCESS;
	gmmsp_instrRange instrPtr;    
    
    if (gmmsp_invalidViInt32Range (channel, 1, 8))
    	return VI_ERROR_PARAMETER2;
    if (gmmsp_invalidViBooleanRange (state))	 
       return VI_ERROR_PARAMETER3;

    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr))
  	Delay (instrPtr->ioDelay);
	CHECKERR (viPrintf (instrSession, "OU %ld,%s", channel, StateArr[state]));
    CHECKERR (gmmsp_checkStatus (instrSession));
    
	return gmmsp_status;
}

/*===========================================================================*/
/* Function: Get Output State                                                */
/* Purpose:  This function queries power output state.						 */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_confGetOutputState (ViSession instrSession,
										    ViInt32 channel,
                                            ViPBoolean state)
{                                         
    ViStatus	gmmsp_status = VI_SUCCESS;

	if (gmmsp_invalidViInt32Range (channel, 1, 8))
    	return VI_ERROR_PARAMETER2;
 

	CHECKERR (gmmsp_readChannelState (instrSession, "OU?", channel, state));
    CHECKERR (gmmsp_checkStatus (instrSession));
	
	return gmmsp_status;
}           

/*===========================================================================*/
/* Function: Set Outp Foldback Mode Setting                                  */
/* Purpose:  This function sets the operating mode of the output.			 */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_confSetOutputMode (ViSession instrSession,
										   ViInt32 channel,
                                           ViInt32 mode)
{                                         
    ViStatus	gmmsp_status = VI_SUCCESS;
	gmmsp_instrRange instrPtr;    
    
    if (gmmsp_invalidViInt32Range (channel, 1, 8))
    	return VI_ERROR_PARAMETER2;
	if (gmmsp_invalidViInt32Range (mode, 0, 1))	 
       return VI_ERROR_PARAMETER3;

    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr))
  	Delay (instrPtr->ioDelay);
	CHECKERR (viPrintf (instrSession, "MO %ld,%s", channel, OutputModeArr[mode]));
    CHECKERR (gmmsp_checkStatus (instrSession));
    
	return gmmsp_status;
}

/*===========================================================================*/
/* Function: Get Outp Foldback Mode Setting                                  */
/* Purpose:  This function queries the operating mode of the output.         */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_confGetOutputMode (ViSession instrSession,
										   ViInt32 channel,
                                           ViPInt32 mode)
{                                         
    ViStatus	gmmsp_status = VI_SUCCESS;
    ViChar		tmpBuffer[BUFFER_SIZE],
    			rdBuffer[256],
    			*p2buffer,
      			smode[256];
	gmmsp_instrRange instrPtr;    
    
    if (gmmsp_invalidViInt32Range (channel, 1, 8))
    	return VI_ERROR_PARAMETER2;
    if (!mode)	 
       return VI_ERROR_PARAMETER3;

  	CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
  	Delay (instrPtr->ioDelay);
  	CHECKERR (viPrintf (instrSession, "MO?"));
  	Delay (instrPtr->ioDelay);
  	CHECKERR (viScanf (instrSession, "%256[^\n]", rdBuffer));
  	
    sprintf(tmpBuffer, "%ld,", channel);
    if ((p2buffer = strstr(rdBuffer, tmpBuffer)) != VI_NULL)
    	{
    	p2buffer += 2;
    	sscanf(p2buffer, "%256[^,]", smode);
    	}
    else
    	return VI_ERROR_INSTR_INTERPRETING_RESPONSE;
    	
    *mode = (ViInt32) (strcmp( smode, "CV") ? 1 : 0);

    CHECKERR (gmmsp_checkStatus (instrSession));
    
	return gmmsp_status;
}

/*===========================================================================*/
/* Function: Output Foldback Mode Actual                                     */
/* Purpose:  This function queries the actual operating mode of the output.  */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_confOutpFoldModAct (ViSession instrSession,
                                            ViInt32 channel,
                                            ViPInt32 foldbackModeAct)
{                               
    ViStatus	gmmsp_status = VI_SUCCESS;
	gmmsp_instrRange instrPtr;
	ViInt32		value1,value2;
    
    if (gmmsp_invalidViInt32Range (channel, 1, 8))
    	return VI_ERROR_PARAMETER2;
    if (!foldbackModeAct)	 
       return VI_ERROR_PARAMETER3;
    
	CHECKERR (gmmsp_utilGetReg (instrSession, 2, &value1));       
    CHECKERR (gmmsp_utilGetReg (instrSession, 5, &value2));   
   
   		if (1<<(channel-1) && value1)  
   			*foldbackModeAct = 2;
   		else
   			if (1<<(channel-1) && value2)  	
   				*foldbackModeAct = 1;  
   			else
   				*foldbackModeAct = 0;
   				
    CHECKERR (gmmsp_checkStatus (instrSession));
    
	return gmmsp_status;
}

/*===========================================================================*/
/* Function: Set Source State                                                */
/* Purpose:  This function selects source ON/OFF.							 */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_confSetSourceState (ViSession instrSession,
										    ViInt32 channel,
                                            ViBoolean state)
{                               
    ViStatus	gmmsp_status = VI_SUCCESS;
	gmmsp_instrRange instrPtr;    
    
    if (gmmsp_invalidViInt32Range (channel, 1, 8))
    	return VI_ERROR_PARAMETER2;
    if (gmmsp_invalidViBooleanRange (state))	 
       return VI_ERROR_PARAMETER3;
       
    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr))
  	Delay (instrPtr->ioDelay);
	CHECKERR (viPrintf (instrSession, "SO %ld,%s", channel, StateArr[state]));
    CHECKERR (gmmsp_checkStatus (instrSession));
    
	return gmmsp_status;
}

/*===========================================================================*/
/* Function: Get Source State                                                */
/* Purpose:  This function queries source state. 							 */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_confGetSourceState (ViSession instrSession,
										    ViInt32 channel,
                                            ViPBoolean state)
{                                         
    ViStatus	gmmsp_status = VI_SUCCESS;

	if (gmmsp_invalidViInt32Range (channel, 1, 8))
    	return VI_ERROR_PARAMETER2;
    if (!state)	 
       return VI_ERROR_PARAMETER3;

	CHECKERR (gmmsp_readChannelState (instrSession, "SO?", channel, state));
    CHECKERR (gmmsp_checkStatus (instrSession));
	
	return gmmsp_status;
}           

/*===========================================================================*/
/* Function: Set Sink State                                                  */
/* Purpose:  This function selects sink ON/OFF.								 */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_confSetSinkState (ViSession instrSession,
								    	  ViInt32 channel,
                                          ViBoolean state)
{                               
    ViStatus	gmmsp_status = VI_SUCCESS;
	gmmsp_instrRange instrPtr;    
    
    if (gmmsp_invalidViInt32Range (channel, 1, 8))
    	return VI_ERROR_PARAMETER2;
    if (gmmsp_invalidViBooleanRange (state))	 
       return VI_ERROR_PARAMETER3;
    
    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr))
  	Delay (instrPtr->ioDelay);
	CHECKERR (viPrintf (instrSession, "SI %ld,%s", channel, StateArr[state]));
    CHECKERR (gmmsp_checkStatus (instrSession));
    
	return gmmsp_status;
}

/*===========================================================================*/
/* Function: Get Sink State                                                  */
/* Purpose:  This function queries sink state.								 */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_confGetSinkState (ViSession instrSession,
										  ViInt32 channel,
                                          ViPBoolean state)
{                                         
    ViStatus	gmmsp_status = VI_SUCCESS;

	if (gmmsp_invalidViInt32Range (channel, 1, 8))
    	return VI_ERROR_PARAMETER2;
    if (!state)	 
       return VI_ERROR_PARAMETER3;

	CHECKERR (gmmsp_readChannelState (instrSession, "SI?", channel, state));
	CHECKERR (gmmsp_checkStatus (instrSession));
	
	return gmmsp_status;
}

/*===========================================================================*/
/* Function: Set Output Foldback Delay                                       */
/* Purpose:  This function defines the foldback delay.						 */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_confSetOCPDelay (ViSession instrSession,
										 ViInt32 channel,
                                         ViReal64 OCPDelay)
{                               
    ViStatus	gmmsp_status = VI_SUCCESS;
	gmmsp_instrRange	instrPtr;    
    
    if (gmmsp_invalidViInt32Range (channel, 1, 8))
    	return VI_ERROR_PARAMETER2;
    if (gmmsp_invalidViReal64Range (OCPDelay, 0.0, 9.999))	 
       return VI_ERROR_PARAMETER3;
    
    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
  	Delay (instrPtr->ioDelay);
	CHECKERR (viPrintf (instrSession, "DE %ld,%.3f", channel, OCPDelay));
    CHECKERR (gmmsp_checkStatus (instrSession));
    
	return gmmsp_status;
}

/*===========================================================================*/
/* Function: Get Output Foldback Delay                                       */
/* Purpose:  This function queries the foldback delay. 						 */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_confGetOCPDelay (ViSession instrSession,
										 ViInt32 channel,
                                         ViPReal64 OCPDelay)
{                               
    ViStatus	gmmsp_status = VI_SUCCESS;

    if (gmmsp_invalidViInt32Range (channel, 1, 8))
    	return VI_ERROR_PARAMETER2;
    if (!OCPDelay)	 
       return VI_ERROR_PARAMETER3;
  
   	CHECKERR (gmmsp_readChannelValue (instrSession, "DE?", channel, OCPDelay));
    CHECKERR (gmmsp_checkStatus (instrSession));

	return gmmsp_status;
}

/*===========================================================================*/
/* Function: Set Output Foldback State                                       */
/* Purpose:  This function selects foldback ON/OFF.							 */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_confSetOCPState (ViSession instrSession,
										 ViInt32 channel,
                                         ViBoolean state)
{                               
    ViStatus	gmmsp_status = VI_SUCCESS;
	gmmsp_instrRange	instrPtr;    
    
    if (gmmsp_invalidViInt32Range (channel, 1, 8))
    	return VI_ERROR_PARAMETER2;
    if (gmmsp_invalidViBooleanRange (state))	 
       return VI_ERROR_PARAMETER3;
    
    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
  	Delay (instrPtr->ioDelay);
	CHECKERR (viPrintf (instrSession, "FO %ld,%s", channel, StateArr[state]));
    CHECKERR (gmmsp_checkStatus (instrSession));
    
	return gmmsp_status;
}

/*===========================================================================*/
/* Function: Get Output Foldback State                                       */
/* Purpose:  This function queries foldback state.							 */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_confGetOCPState (ViSession instrSession,
										 ViInt32 channel,
                                         ViPBoolean state)
{                                         
    ViStatus	gmmsp_status = VI_SUCCESS;

	if (gmmsp_invalidViInt32Range (channel, 1, 8))
    	return VI_ERROR_PARAMETER2;
    if (!state)	 
       return VI_ERROR_PARAMETER3;

	CHECKERR (gmmsp_readChannelState (instrSession, "FO?", channel, state));
    CHECKERR (gmmsp_checkStatus (instrSession));
    
	return gmmsp_status;
}

/*===========================================================================*/
/* Function: Set Group Configure                                             */
/* Purpose:  This function sets the channels to group. 						 */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_confSetGroup (ViSession instrSession,
									  ViInt32 group)
{                               
    ViStatus	gmmsp_status = VI_SUCCESS;
    ViChar 		tmpBuffer[BUFFER_SIZE];
    ViInt32		length, i;
    
    if (gmmsp_invalidViInt32Range (group, 0, 255))
    	return VI_ERROR_PARAMETER2;
  	
	CHECKERR (gmmsp_writeGroup (instrSession, group, "SET"));
    CHECKERR (gmmsp_checkStatus (instrSession));
    
	return gmmsp_status;
}

/*===========================================================================*/
/* Function: Get Group Configure                                             */
/* Purpose:  This function queries channels in groups.						 */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_confGetGroup (ViSession instrSession,
									  ViPInt32 group)
{                               
    ViStatus	gmmsp_status = VI_SUCCESS;
    ViInt32		value;
    ViInt32		i;
    ViChar		rdBuffer[256],
    			*p2buffer;

    if (!group)	 
       return VI_ERROR_PARAMETER2;
       
    CHECKERR (gmmsp_readGroup (instrSession, group, "SET?"));
    CHECKERR (gmmsp_checkStatus (instrSession));

	return gmmsp_status;
}

/*===========================================================================*/
/* Function: Set Group State                                                 */
/* Purpose:  This function selects group ON/OFF.							 */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_confSetGroupState (ViSession instrSession,
                                           ViBoolean state)
{                               
    ViStatus	gmmsp_status = VI_SUCCESS;
	gmmsp_instrRange instrPtr;    
    
    if (gmmsp_invalidViBooleanRange (state))	 
       return VI_ERROR_PARAMETER2;

    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr))
  	Delay (instrPtr->ioDelay);
	CHECKERR (viPrintf (instrSession, "GR %s", StateArr[state]));
    CHECKERR (gmmsp_checkStatus (instrSession));
    
	return gmmsp_status;
}

/*===========================================================================*/
/* Function: Get Group State                                                 */
/* Purpose:  This function queries group state.								 */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_confGetGroupState (ViSession instrSession,
                                           ViPBoolean state)
{                                         
    ViStatus	gmmsp_status = VI_SUCCESS;
    ViChar		rdBuffer[BUFFER_SIZE];
	ViInt32		rdBufferSize = sizeof(rdBuffer);
	gmmsp_instrRange instrPtr;    

    if (!state)	 
       return VI_ERROR_PARAMETER2;

  	CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
  	Delay (instrPtr->ioDelay);
  	CHECKERR (viPrintf (instrSession, "GR?"));
  	Delay (instrPtr->ioDelay);
  	CHECKERR (viScanf (instrSession, "%*[^ ] %#s", &rdBufferSize, rdBuffer));	
  	
    *state = (ViBoolean) (strstr( rdBuffer, "ON") ? VI_TRUE : VI_FALSE);
    CHECKERR (gmmsp_checkStatus (instrSession));
	
	return gmmsp_status;
}

/*===========================================================================*/
/* Function: Set Measure Group                                               */
/* Purpose:  This function sets the channels to measure group. 				 */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_confSetMeasGroup (ViSession instrSession,
									      ViInt32 group)
{                               
    ViStatus	gmmsp_status = VI_SUCCESS;
    ViChar 		tmpBuffer[BUFFER_SIZE];
    ViInt32		length, i;
	
    if (gmmsp_invalidViInt32Range (group, 0, 255))
    	return VI_ERROR_PARAMETER2;
    		     	
	CHECKERR (gmmsp_writeGroup (instrSession, group, "ME"));
    CHECKERR (gmmsp_checkStatus (instrSession));

	return gmmsp_status;
}

/*===========================================================================*/
/* Function: Get Measure Group                                               */
/* Purpose:  This function queries channels in groups.	    				 */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_confGetMeasGroup (ViSession instrSession,
								    	  ViPInt32 group)
{                               
    ViStatus	gmmsp_status = VI_SUCCESS;
    ViInt32		value;
    ViInt32		i;
    ViChar		rdBuffer[256],
    			*p2buffer;

    if (!group)	 
       return VI_ERROR_PARAMETER2;

    CHECKERR (gmmsp_readGroup (instrSession, group, "ME?"));
    CHECKERR (gmmsp_checkStatus (instrSession));

	return gmmsp_status;
}

/*===========================================================================*/
/* Function: Set MinMax State                                                */
/* Purpose:  This function selects MINMAX mode.								 */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_confSetMinMaxState (ViSession instrSession,
                                            ViBoolean state)
{                                            
    ViStatus	gmmsp_status = VI_SUCCESS;
	gmmsp_instrRange instrPtr;    
    
    if (gmmsp_invalidViBooleanRange (state))	 
       return VI_ERROR_PARAMETER2;
    
    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr))
  	Delay (instrPtr->ioDelay);
	CHECKERR (viPrintf (instrSession, "MI %s", StateArr[state]));
    CHECKERR (gmmsp_checkStatus (instrSession));

	return gmmsp_status;
}

/*===========================================================================*/
/* Function: Get MinMax State                                                */
/* Purpose:  This function queries MINMAX mode. 							 */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_confGetMinMaxState (ViSession instrSession,
                                             ViPBoolean state)
{                                         
    ViStatus	gmmsp_status = VI_SUCCESS;
    ViChar 		mode[BUFFER_SIZE];
	gmmsp_instrRange instrPtr;    

    if (!state)	 
       return VI_ERROR_PARAMETER2;
       
    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr))
  	Delay (instrPtr->ioDelay);
	CHECKERR (viPrintf (instrSession, "MI?"));
  	Delay (instrPtr->ioDelay);
	CHECKERR (viScanf (instrSession, "%*[^ ] %s", mode));
	
    *state = (ViBoolean) (strstr( mode, "ON") ? VI_TRUE : VI_FALSE);
    CHECKERR (gmmsp_checkStatus (instrSession));

	return gmmsp_status;
}

/*===========================================================================*/
/* Function: Set Power On Mode                                               */
/* Purpose:  This function defines the status of the device setting after	 */
/*			 power on.								                         */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_confSetPwrOnMode (ViSession instrSession,
                                          ViInt32 powerMode)
{                               
    ViStatus gmmsp_status = VI_SUCCESS;
	gmmsp_instrRange instrPtr;    
    
    if (gmmsp_invalidViInt32Range (powerMode,0,2))	 
       return VI_ERROR_PARAMETER2;
       
    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr))
  	Delay (instrPtr->ioDelay);
	CHECKERR (viPrintf (instrSession, "POW %s", PowerModeArr[powerMode]));
    CHECKERR (gmmsp_checkStatus (instrSession));

	return gmmsp_status;
}

/*===========================================================================*/
/* Function: Get Power On Mode                                               */
/* Purpose:  This function queries the status of the device setting after	 */
/*			 power on.														 */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_confGetPwrOnMode (ViSession instrSession,
                                           ViPInt32 powerMode)
{                                         
    ViStatus gmmsp_status = VI_SUCCESS;
    ViChar mode[BUFFER_SIZE];
    ViInt32 i;
	gmmsp_instrRange instrPtr;    
    
    if (!powerMode)	 
       return VI_ERROR_PARAMETER2;
       
    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr))
  	Delay (instrPtr->ioDelay);
	CHECKERR (viPrintf (instrSession, "POW?"));
	CHECKERR (viScanf (instrSession, "%*[^ ] %s", mode));

	for (i = 0; PowerModeArr[i]; i++)
		if (strstr(mode, PowerModeArr[i])) break;

	if (!PowerModeArr[i])
		return VI_ERROR_INSTR_INTERPRETING_RESPONSE;

    *powerMode = i;    
    CHECKERR (gmmsp_checkStatus (instrSession));
    
	return gmmsp_status;
}                                       

/*===========================================================================*/
/* Function: Reset MinMax Mode                                               */
/* Purpose:  This function resets values in MinMax memory.   				 */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_resetMinMax (ViSession instrSession)
{                                         
    ViStatus gmmsp_status = VI_SUCCESS;
	gmmsp_instrRange instrPtr;    

    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr))
  	Delay (instrPtr->ioDelay);    
	CHECKERR (viPrintf (instrSession, "MI RST"));
    CHECKERR (gmmsp_checkStatus (instrSession));

	return gmmsp_status;
}                          

/*===========================================================================*/
/* Function: Recall Register                                                 */
/* Purpose:  This function recalls the specified register from the			 */
/*			 instrument's memory.              								 */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_dataRecallRegister (ViSession instrSession,
                                            ViInt32 registerNo)
{
    ViStatus gmmsp_status = VI_SUCCESS;
	gmmsp_instrRange instrPtr;    

    if (gmmsp_invalidViInt32Range (registerNo, 1, 9))	 
       return VI_ERROR_PARAMETER2;
       
    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr))
  	Delay (instrPtr->ioDelay);
	CHECKERR (viPrintf (instrSession, "*RCL %ld", registerNo));
    CHECKERR (gmmsp_checkStatus (instrSession));

	return gmmsp_status;
}

/*===========================================================================*/
/* Function: Save Register                                                   */
/* Purpose:  This function stores settings to the register in instrument's   */
/* 			 memory.						 								 */												 
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_dataSaveRegister (ViSession instrSession,
                                          ViInt32 registerNo)
{                               
    ViStatus gmmsp_status = VI_SUCCESS;
	gmmsp_instrRange instrPtr;    

    if (gmmsp_invalidViInt32Range (registerNo, 1, 9))	 
       return VI_ERROR_PARAMETER2;
       
    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr))
  	Delay (instrPtr->ioDelay);
	CHECKERR (viPrintf (instrSession, "*SAV %ld", registerNo));
    CHECKERR (gmmsp_checkStatus (instrSession));

	return gmmsp_status;
}

/*===========================================================================*/
/* Function: Fetch MinMax                                                    */
/* Purpose:  This function returns the limit values of the output current and*/
/* 			 voltage measured and saved in MINMAX memory during MINMAX ON.	 */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_fetchMinMax (ViSession instrSession,ViInt32 channel,
                                     ViPReal64 minVoltage,ViPReal64 maxVoltage,
                                     ViPReal64 minCurrent,ViPReal64 maxCurrent)
{                                         
    ViStatus gmmsp_status = VI_SUCCESS;

	if (gmmsp_invalidViInt32Range (channel, 1, 8))
    	return VI_ERROR_PARAMETER2;

    if (minVoltage)	
    	CHECKERR (gmmsp_readChannelValue (instrSession, "UMIN?", channel, minVoltage));
    if (maxVoltage)	 
    	CHECKERR (gmmsp_readChannelValue (instrSession, "UMAX?", channel, maxVoltage));
    if (minCurrent)	 
    	CHECKERR (gmmsp_readChannelValue (instrSession, "IMIN?", channel, minCurrent));
    if (maxCurrent)	 
    	CHECKERR (gmmsp_readChannelValue (instrSession, "IMAX?", channel, maxCurrent));

    CHECKERR (gmmsp_checkStatus (instrSession));

	return gmmsp_status;
}

/*===========================================================================*/
/* Function: Meas Output Voltage                                             */
/* Purpose:  This function queries the actual measured value of the output   */
/*			 voltage.	   						 						     */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_measOutVolt (ViSession instrSession,
									 ViInt32 channel,
									 ViPReal64 voltage, ViBoolean checkGroup,
									 ViPBoolean validVal)
{                                         
    ViStatus gmmsp_status = VI_SUCCESS;
    ViInt32 group;

	if (gmmsp_invalidViInt32Range (channel, 1, 8))
    	return VI_ERROR_PARAMETER2;
    if (!voltage)	 
       return VI_ERROR_PARAMETER3;
	if (gmmsp_invalidViBooleanRange (checkGroup))
    	return VI_ERROR_PARAMETER4;
       
	if (checkGroup) {
		CHECKERR (gmmsp_confGetMeasGroup (instrSession, &group));
		if ((1<<(channel-1) && group) >= 0 ) { 
   			CHECKERR (gmmsp_readChannelValue (instrSession, "UOUT?", channel, voltage)); 
   			*validVal = 1;
		}
		
		else {
			*validVal = 0;
			*voltage = 0.0;
		}
	}
	else 
		CHECKERR (gmmsp_readChannelValue (instrSession, "UOUT?", channel, voltage)); 
	
    CHECKERR (gmmsp_checkStatus (instrSession));
    
	return gmmsp_status;
}                                    

/*===========================================================================*/
/* Function: Meas Output Current                                             */
/* Purpose:  This function queries the actual measured value of the output   */
/*			 current.	 													 */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_measOutCurr (ViSession instrSession,
									 ViInt32 channel, ViPReal64 current,
									 ViBoolean checkGroup,
									 ViPBoolean validVal)
{                               
    ViStatus gmmsp_status = VI_SUCCESS;
    ViInt32 group;
    
	if (gmmsp_invalidViInt32Range (channel, 1, 8))
    	return VI_ERROR_PARAMETER2;
    if (!current)	 
       return VI_ERROR_PARAMETER3;
	if (gmmsp_invalidViBooleanRange (checkGroup))
    	return VI_ERROR_PARAMETER4;
       
	if (checkGroup) {
		CHECKERR (gmmsp_confGetMeasGroup (instrSession, &group));
		if ((1<<(channel-1) && group) >= 0 ) { 
   			CHECKERR (gmmsp_readChannelValue (instrSession, "IOUT?", channel, current)); 
   			*validVal = 1;
		}
		
		else {
			*validVal = 0;
			*current = 0.0;
		}
	}
	else 
		CHECKERR (gmmsp_readChannelValue (instrSession, "IOUT?", channel, current)); 
		
	CHECKERR (gmmsp_checkStatus (instrSession));
    
	return gmmsp_status;
}

/*===========================================================================*/
/* Function: Meas Output Power                                               */
/* Purpose:  This function queries the actually measured power on the output.*/
/*			 The time required to acquire and process the reading is approx. */
/*			 90 ms.							 								 */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_measOutPower (ViSession instrSession,
									   	 ViInt32 channel, ViPReal64 powerOut,
									   	 ViBoolean checkGroup,
									 	 ViPBoolean validVal)
{                               
    ViStatus gmmsp_status = VI_SUCCESS;
    ViInt32 group;
    
	if (gmmsp_invalidViInt32Range (channel, 1, 8))
    	return VI_ERROR_PARAMETER2;
    if (!powerOut)	 
       return VI_ERROR_PARAMETER3;
	if (gmmsp_invalidViBooleanRange (checkGroup))
    	return VI_ERROR_PARAMETER4;
    
       
	if (checkGroup) {
		CHECKERR (gmmsp_confGetMeasGroup (instrSession, &group));
		if ((1<<(channel-1) && group) >= 0 ) { 
   			CHECKERR (gmmsp_readChannelValue (instrSession, "POUT?", channel, powerOut)); 
   			*validVal = 1;
		}
		
		else {
			*validVal = 0;
			*powerOut = 0.0;
		}
	}
	else 
		CHECKERR (gmmsp_readChannelValue (instrSession, "POUT?", channel, powerOut)); 
	
    CHECKERR (gmmsp_checkStatus (instrSession));
    
	return gmmsp_status;																	
}

/*===========================================================================*/
/* Function:Meas All Output Voltage                                          */
/* Purpose: This function queries the actual measured voltage of the all	 */
/*			channels, but these channels have to be in the measurement group.*/			 
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_measAllOutVolt (ViSession instrSession, ViBoolean setGroup,
									    ViPReal64 voltage1,ViPReal64 voltage2,
									    ViPReal64 voltage3,ViPReal64 voltage4,
									    ViPReal64 voltage5,ViPReal64 voltage6,
									    ViPReal64 voltage7,ViPReal64 voltage8)
{                                         
    ViStatus	gmmsp_status = VI_SUCCESS;
    ViChar		rdBuffer[256],
    			*p2buffer;
	ViInt32  	group, oldGroup;    			
	gmmsp_instrRange instrPtr;    
    
    if (gmmsp_invalidViBooleanRange (setGroup))
    	return VI_ERROR_PARAMETER2;
    
    if (setGroup) {
    	CHECKERR (gmmsp_confGetMeasGroup (instrSession, &oldGroup));
    	CHECKERR (gmmsp_confSetMeasGroup (instrSession, 255)); //set to all channels
    }
    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
  	Delay (instrPtr->ioDelay);
	CHECKERR (viPrintf (instrSession, "UOUT?"));
  	Delay (instrPtr->ioDelay);
	CHECKERR (viScanf (instrSession, "%256[^\n]", rdBuffer));

	if (voltage1)
		{
		p2buffer = strstr(rdBuffer," 1,") + 3;
 		sscanf(p2buffer, "%Lf", voltage1);
 		}
	if (voltage2)	
		{
		p2buffer = strstr(rdBuffer,",2,") + 3;
 		sscanf(p2buffer, "%Lf", voltage2);
 		}
	if (voltage3)	
		{
		p2buffer = strstr(rdBuffer,",3,") + 3;
 		sscanf(p2buffer, "%Lf", voltage3);
 		}
	if (voltage4)	
		{
		p2buffer = strstr(rdBuffer,",4,") + 3;
 		sscanf(p2buffer, "%Lf", voltage4);
 		}
 	if (voltage5)	
		{
		p2buffer = strstr(rdBuffer,",5,") + 3;
 		sscanf(p2buffer, "%Lf", voltage5);
 		}
	if (voltage6)	
		{
		p2buffer = strstr(rdBuffer,",6,") + 3;
 		sscanf(p2buffer, "%Lf", voltage6);
 		}
	if (voltage7)	
		{
		p2buffer = strstr(rdBuffer,",7,") + 3;
 		sscanf(p2buffer, "%Lf", voltage7);
 		}
	if (voltage8)	
		{
		p2buffer = strstr(rdBuffer,",8,") + 3;
 		sscanf(p2buffer, "%Lf", voltage8);
 		}
    
    if (setGroup) 
    	CHECKERR (gmmsp_confSetMeasGroup (instrSession, oldGroup));
    
    CHECKERR (gmmsp_checkStatus (instrSession));
    
	return gmmsp_status;
}                                    

/*===========================================================================*/
/* Function: Get All Output Current                                          */
/* Purpose: This function queries the actual measured current of the all	 */
/*			channels, but these channels have to be in the measurement group.*/
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_measAllOutCurr (ViSession instrSession, ViBoolean setGroup,
									    ViPReal64 current1,ViPReal64 current2,
									    ViPReal64 current3,ViPReal64 current4,
									    ViPReal64 current5,ViPReal64 current6,
									    ViPReal64 current7,ViPReal64 current8)
{                               
    ViStatus	gmmsp_status = VI_SUCCESS;
    ViChar		rdBuffer[256],
    			*p2buffer;
	ViInt32  	group, oldGroup;
	gmmsp_instrRange instrPtr;    
    
    if (gmmsp_invalidViBooleanRange (setGroup))
    	return VI_ERROR_PARAMETER2;
    
    if (setGroup) {
    	CHECKERR (gmmsp_confGetMeasGroup (instrSession, &oldGroup));
    	CHECKERR (gmmsp_confSetMeasGroup (instrSession, 255)); //set to all channels
    }
    
    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
  	Delay (instrPtr->ioDelay);
	CHECKERR (viPrintf (instrSession, "IOUT?"));
  	Delay (instrPtr->ioDelay);
	CHECKERR (viScanf (instrSession, "%256[^\n]", rdBuffer));    
	
	if (current1)	
		{
		p2buffer = strstr(rdBuffer," 1,") + 3;
 		sscanf(p2buffer, "%Lf", current1);
 		}
	if (current2)	
		{
		p2buffer = strstr(rdBuffer,",2,") + 3;
 		sscanf(p2buffer, "%Lf", current2);
 		}
	if (current3)	
		{
		p2buffer = strstr(rdBuffer,",3,") + 3;
 		sscanf(p2buffer, "%Lf", current3);
 		}
	if (current4)	
		{
		p2buffer = strstr(rdBuffer,",4,") + 3;
 		sscanf(p2buffer, "%Lf", current4);
 		}
	if (current5)	
		{
		p2buffer = strstr(rdBuffer,",5,") + 3;
 		sscanf(p2buffer, "%Lf", current5);
 		}
	if (current6)	
		{
		p2buffer = strstr(rdBuffer,",6,") + 3;
 		sscanf(p2buffer, "%Lf", current6);
 		}
	if (current7)	
		{
		p2buffer = strstr(rdBuffer,",7,") + 3;
 		sscanf(p2buffer, "%Lf", current7);
 		}
	if (current8)	
		{
		p2buffer = strstr(rdBuffer,",8,") + 3;
 		sscanf(p2buffer, "%Lf", current8);
 		}
    
    if (setGroup) 
    	CHECKERR (gmmsp_confSetMeasGroup (instrSession, oldGroup));
    	
    CHECKERR (gmmsp_checkStatus (instrSession));
    
	return gmmsp_status;
}

/*===========================================================================*/
/* Function: Meas All Output Power                                           */
/* Purpose:  This function queries the actually measured power on the output.*/
/*			 The time required to acquire and process the reading is approx. */
/*			 90 ms.				 											 */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_measAllOutPower (ViSession instrSession, ViBoolean setGroup,
									     ViPReal64 power1,ViPReal64 power2,
									     ViPReal64 power3,ViPReal64 power4,
									     ViPReal64 power5,ViPReal64 power6,
									     ViPReal64 power7,ViPReal64 power8)
{                               
    ViStatus	gmmsp_status = VI_SUCCESS;
    ViChar		rdBuffer[256],
    			*p2buffer;
	ViInt32  	group, oldGroup;
	gmmsp_instrRange instrPtr;    
    
    if (gmmsp_invalidViBooleanRange (setGroup))
    	return VI_ERROR_PARAMETER2;
    
    if (setGroup) {
    	CHECKERR (gmmsp_confGetMeasGroup (instrSession, &oldGroup));
    	CHECKERR (gmmsp_confSetMeasGroup (instrSession, 255)); //set to all channels
    }
    
    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
  	Delay (instrPtr->ioDelay);
	CHECKERR (viPrintf (instrSession, "POUT?"));
  	Delay (instrPtr->ioDelay);
	CHECKERR (viScanf (instrSession, "%256[^\n]", rdBuffer));

	if (power1)
		{
		p2buffer = strstr(rdBuffer," 1,") + 3;
 		sscanf(p2buffer, "%Lf", power1);
 		}
	if (power2)
		{
		p2buffer = strstr(rdBuffer,",2,") + 3;
 		sscanf(p2buffer, "%Lf", power2);
 		}
	if (power3)
		{
		p2buffer = strstr(rdBuffer,",3,") + 3;
 		sscanf(p2buffer, "%Lf", power3);
 		}
	if (power4)
		{
		p2buffer = strstr(rdBuffer,",4,") + 3;
 		sscanf(p2buffer, "%Lf", power4);
 		}
	if (power5)
		{
		p2buffer = strstr(rdBuffer,",5,") + 3;
 		sscanf(p2buffer, "%Lf", power5);
 		}
	if (power6)
		{
		p2buffer = strstr(rdBuffer,",6,") + 3;
 		sscanf(p2buffer, "%Lf", power6);
 		}
	if (power7)
		{
		p2buffer = strstr(rdBuffer,",7,") + 3;
 		sscanf(p2buffer, "%Lf", power7);
 		}
	if (power8)
		{
		p2buffer = strstr(rdBuffer,",8,") + 3;
 		sscanf(p2buffer, "%Lf", power8);
 		}

    if (setGroup) 
    	CHECKERR (gmmsp_confSetMeasGroup (instrSession, oldGroup));
    	
    CHECKERR (gmmsp_checkStatus (instrSession));

	return gmmsp_status;
}

/*****************************************************************************/
/*-------- INSERT USER-CALLABLE INSTRUMENT-DEPENDENT ROUTINES HERE ----------*/
/*****************************************************************************/
/*===========================================================================*/
/* Function: State Checking                                                  */
/* Purpose:  This function switches ON/OFF state checking of the instrument  */
/*           (reading of the Standard Event Register and checking it for     */
/*           error). Nearly all driver function are using state checking.    */
/*           Switch this function to ON when debug your application. For     */
/*           better bus throughput and instruments performance switch it OFF.*/
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_errorCheckState (ViSession instrSession,
                                         ViBoolean state)
{
    ViStatus	gmmsp_status = VI_SUCCESS;
    gmmsp_instrRange instrPtr;

    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));

    instrPtr -> errorChecking = state;

    return gmmsp_status;
}

/*===========================================================================*/
/* Function: Write To Instrument                                             */
/* Purpose:  This function writes a command string to the instrument.        */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_writeInstrData (ViSession instrSession, 
										ViString writeBuffer)
{
    ViStatus gmmsp_status = VI_SUCCESS;
	gmmsp_instrRange instrPtr;    
    
    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
  	Delay (instrPtr->ioDelay);
    CHECKERR (viPrintf (instrSession, "%s", writeBuffer));
    
    return gmmsp_status;
}

/*===========================================================================*/
/* Function: Read Instrument Buffer                                          */
/* Purpose:  This function reads the output buffer of the instrument.        */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_readInstrData (ViSession instrSession,
                                       ViInt32 numberBytesToRead,
                                       ViChar _VI_FAR readBuffer[],
                                       ViPInt32 numBytesRead)
{
    ViStatus	gmmsp_status = VI_SUCCESS;
    ViUInt32	tmpCnt;
    gmmsp_instrRange instrPtr;    

    if (numBytesRead)
    	*numBytesRead = 0L;
    
    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
  	Delay (instrPtr->ioDelay);    
    CHECKERR (viRead (instrSession, (ViBuf)readBuffer, numberBytesToRead, &tmpCnt));
    
    if (numBytesRead)
    	*numBytesRead = (ViInt32)tmpCnt;
    
    return gmmsp_status;
}

/*===========================================================================*/
/* Function: Reset                                                           */
/* Purpose:  This function resets the instrument.                            */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_reset (ViSession instrSession)
{
    ViStatus	gmmsp_status = VI_SUCCESS;
	gmmsp_instrRange instrPtr;    			

    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr))
    /*  Initialize the instrument to a known state.  */
    Delay(instrPtr->ioDelay);
    CHECKERR (viPrintf (instrSession, "*RST"));
    CHECKERR (gmmsp_defaultInstrSetup (instrSession));
        
    return gmmsp_status;
}

/*===========================================================================*/
/* Function: Self-Test                                                       */
/* Purpose:  This function executes the instrument self-test and returns     */
/*           the result.			                                         */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_self_test (ViSession instrSession,
                                   ViPInt16 testResult,
                                   ViChar _VI_FAR testMessage[])
{
    ViStatus	gmmsp_status = VI_SUCCESS;
	gmmsp_instrRange instrPtr;    			

	if (!testResult)
    	 return VI_ERROR_PARAMETER2;		
    	 
    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr))
    Delay(instrPtr->ioDelay);
    CHECKERR (viPrintf (instrSession, "*TST?"));
    Delay(instrPtr->ioDelay);
    CHECKERR (viScanf (instrSession, "%hd", testResult));

	if (testMessage)
		strcpy(testMessage, (testResult) ? "Self-test passed" : "Self-test failed");

    return gmmsp_status;
}

/*===========================================================================*/
/* Function: Error Query                                                     */
/* Purpose:  This function queries the instrument error queue, and returns   */
/*           the result. If the error query function is not supported by the */
/*           instrument, this function returns the warning                   */
/*           VI_WARN_NSUP_ERROR_QUERY.                                       */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_error_query (ViSession instrSession,
                                     ViPInt32 errorCode,
                                     ViChar _VI_FAR errorMessage[])
{
    return VI_WARN_NSUP_ERROR_QUERY;
}

/*===========================================================================*/
/* Function: Error Message                                                   */
/* Purpose:  This function translates the error return value from the        */
/*           instrument driver into a user-readable string.                  */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_error_message (ViSession instrSession,
                                       ViStatus statusCode,
                                       ViChar _VI_FAR errMessage[])
{
    ViStatus	gmmsp_status = VI_SUCCESS;
    ViInt16		i;

    static gmmsp_tStringValPair statusDescArray[] = {
        {VI_WARN_NSUP_ERROR_QUERY,  "WARNING: Error Query not supported"},     
        {VI_ERROR_PARAMETER1,   "ERROR: Parameter 1 out of range"},
        {VI_ERROR_PARAMETER2,   "ERROR: Parameter 2 out of range"},
        {VI_ERROR_PARAMETER3,   "ERROR: Parameter 3 out of range"},
        {VI_ERROR_FAIL_ID_QUERY,"ERROR: Identification query failed"},
        {VI_ERROR_INV_RESPONSE, "ERROR: Interpreting instrument response"},
        {VI_ERROR_INSTR_INTERPRETING_RESPONSE, "ERROR: Interpreting the instrument's response"},
		{GMMSP_ERROR_INSTRUMENT_ERROR,  "ERROR: Instrument status error"},
        
        {VI_NULL, VI_NULL}
    };

    gmmsp_status = viStatusDesc (instrSession, statusCode, errMessage);
    if (gmmsp_status == VI_WARN_UNKNOWN_STATUS)
    	{
        for (i = 0; statusDescArray[i].stringName; i++)
        	{
            if (statusDescArray[i].stringVal == statusCode)
            	{
                sprintf (errMessage, "%s", statusDescArray[i].stringName);
                return (VI_SUCCESS);
            	}
        	}
        sprintf (errMessage, "Unknown Error 0x%X", statusCode);
        return (VI_WARN_UNKNOWN_STATUS);
    	}
    
    gmmsp_status = VI_SUCCESS;
    return gmmsp_status;
}

/*===========================================================================*/
/* Function: Revision Query                                                  */
/* Purpose:  This function returns the driver and instrument revisions.      */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_revision_query (ViSession instrSession,
                    					ViChar _VI_FAR driverRev[], 
                    					ViChar _VI_FAR instrRev[])
{
    ViStatus	gmmsp_status = VI_SUCCESS;
	gmmsp_instrRange instrPtr;    
    
    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
    
	if (instrRev)
	  	Delay (instrPtr->ioDelay); 
    	CHECKERR (viPrintf (instrSession, "*IDN?"));
	  	Delay (instrPtr->ioDelay);    
    	CHECKERR (viScanf (instrSession, "%*[^,],%*[^,],%*[^,],%256[^\r\n]", instrRev));
    if (driverRev)	
    	strcpy (driverRev, GMMSP_REVISION);
    
    return gmmsp_status;
}

/*===========================================================================*/
/* Function: Set System Register                                             */
/* Purpose:  This function sets specified value to register.				 */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_utilSetReg (ViSession instrSession,
                                    ViInt32 registerName, 
                                    ViInt32 value)
{
    ViStatus	gmmsp_status = VI_SUCCESS;
	gmmsp_instrRange instrPtr;    
    
    if (gmmsp_invalidViInt32Range (registerName, 0, 6))	 
       return VI_ERROR_PARAMETER2;
    if (gmmsp_invalidViInt32Range (value, 0, 255))	 
       return VI_ERROR_PARAMETER2;

    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
	Delay (instrPtr->ioDelay);
	CHECKERR (viPrintf (instrSession, "%s %ld", SetRegArr[registerName], value));

	return gmmsp_status;
}

/*===========================================================================*/
/* Function: Get System Register			                                 */
/* Purpose:  This function gets specified value from register.				 */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_utilGetReg (ViSession instrSession,
                                    ViInt32 registerName, 
                                    ViPInt32 value)
{
    ViStatus	gmmsp_status = VI_SUCCESS;
	gmmsp_instrRange instrPtr;    

    if (gmmsp_invalidViInt32Range (registerName, 0, 15))	 
       return VI_ERROR_PARAMETER2;

    if (!value)	 
       return VI_ERROR_PARAMETER2;
       
    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
	Delay (instrPtr->ioDelay);
	CHECKERR (viPrintf (instrSession, "%s", GetRegArr[registerName]));
	Delay (instrPtr->ioDelay);
	CHECKERR (viScanf (instrSession, "%ld", value));

	return gmmsp_status;
}

/*===========================================================================*/
/* Function: Set Display State                                               */
/* Purpose:  This function selects display mode. This function enables the   */
/*			 digital display to be signalled OFF if remote control is active.*/
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_utilSetDisplayState (ViSession instrSession,
                                             ViBoolean state)
{                               
    ViStatus	gmmsp_status = VI_SUCCESS;
	gmmsp_instrRange instrPtr;    
    
    if (gmmsp_invalidViBooleanRange (state))	 
       return VI_ERROR_PARAMETER2;
       
    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
	Delay (instrPtr->ioDelay);
	CHECKERR (viPrintf (instrSession, "DI %s", StateArr[state]));
    CHECKERR (gmmsp_checkStatus (instrSession));
    
	return gmmsp_status;
}

/*===========================================================================*/
/* Function: Get Display State                                               */
/* Purpose:  This function queries display mode.							 */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_utilGetDisplayState (ViSession instrSession,
                                             ViPBoolean state)
{                                         
    ViStatus	gmmsp_status = VI_SUCCESS;
    ViChar		mode[BUFFER_SIZE];
	gmmsp_instrRange instrPtr;    

       
    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
	Delay (instrPtr->ioDelay);
	CHECKERR (viPrintf (instrSession, "DI?"));
	Delay (instrPtr->ioDelay);
	CHECKERR (viScanf (instrSession, "%*[^ ] %s", mode));
	
    *state = (ViBoolean) (strcmp( mode, "OFF") ? VI_TRUE : VI_FALSE);
    
    CHECKERR (gmmsp_checkStatus (instrSession));
    
	return gmmsp_status;
}                        

/*===========================================================================*/
/* Function: Diagnostics                                                     */
/* Purpose:  This function runs a set of the instrument's internal diagnostic*/
/*			 routines and returns the results. 								 */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_diag (ViSession instrSession,
                              ViChar _VI_FAR selfTestMessage[])
{
    ViStatus	gmmsp_status = VI_SUCCESS;
	gmmsp_instrRange instrPtr;    
	
    if (!selfTestMessage)
    	 return VI_ERROR_PARAMETER2;

    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
	Delay (instrPtr->ioDelay);
    CHECKERR (viPrintf (instrSession, "HID_TST?"));
   	Delay (instrPtr->ioDelay);
	CHECKERR (viScanf (instrSession, "%[^\n]", selfTestMessage));
	
    CHECKERR (gmmsp_checkStatus (instrSession));
	
    return gmmsp_status;
}

/*===========================================================================*/
/* Function: Set Display Text                                                */
/* Purpose:  This function sets specified text to display.					 */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_utilSetDispText (ViSession instrSession,
                              			 ViString text)
{
    ViStatus	gmmsp_status = VI_SUCCESS;
	gmmsp_instrRange instrPtr;    
    
	if (gmmsp_invalidViString (text, 40))	 
       return VI_ERROR_PARAMETER2;
       
  	CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
	Delay (instrPtr->ioDelay);
	CHECKERR (viPrintf (instrSession, "TE %s", text));
    CHECKERR (gmmsp_checkStatus (instrSession));
    
	return gmmsp_status;
}

/*===========================================================================*/
/* Function: Get Display Text                                                */
/* Purpose:  This function queries displayed text.    						 */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_utilGetDispText (ViSession instrSession,
                              			 ViChar text[])
{
    ViStatus	gmmsp_status = VI_SUCCESS;
	gmmsp_instrRange instrPtr;    
	
    if (!text)
    	 return VI_ERROR_PARAMETER2;
  	CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
	Delay (instrPtr->ioDelay);
    CHECKERR (viPrintf (instrSession, "TE?"));
	Delay (instrPtr->ioDelay);
    CHECKERR (viScanf (instrSession, "%[^\n]", text));    
    CHECKERR (gmmsp_checkStatus (instrSession));
	
    return gmmsp_status;
}

/*===========================================================================*/
/* Function: Close                                                           */
/* Purpose:  This function closes the instrument.                            */
/*===========================================================================*/
ViStatus _VI_FUNC gmmsp_close (ViSession instrSession)
{
    ViStatus	gmmsp_status = VI_SUCCESS;
    ViSession	rmSession;
    gmmsp_instrRange instrPtr;

    CHECKERR( viGetAttribute (instrSession, VI_ATTR_RM_SESSION, &rmSession));
    CHECKERR( viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));

    if (instrPtr != NULL) 
        free (instrPtr);

    gmmsp_status = viClose (instrSession);
    viClose (rmSession);

    return gmmsp_status;
}

/*****************************************************************************/
/*= UTILITY ROUTINES (Non-Exportable Functions) =============================*/
/*****************************************************************************/

/*===========================================================================*/
/* Function: Boolean Value Out Of Range - ViBoolean                          */
/* Purpose:  This function checks a Boolean to see if it is equal to VI_TRUE */
/*           or VI_FALSE. If the value is out of range, the return value is  */
/*           VI_TRUE, otherwise the return value is VI_FALSE.                */
/*===========================================================================*/
ViBoolean gmmsp_invalidViBooleanRange (ViBoolean val)
{
    return (ViBoolean)((val != VI_FALSE && val != VI_TRUE) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Long Signed Integer Value Out Of Range - ViInt32                */
/* Purpose:  This function checks a long signed integer value to see if it   */  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range, the return value is VI_TRUE, otherwise the return     */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean gmmsp_invalidViInt32Range (ViInt32 val, ViInt32 min, ViInt32 max)
{
    return (ViBoolean)((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Real (Double) Value Out Of Range - ViReal64                     */
/* Purpose:  This function checks a real (double) value to see if it lies    */  
/*           between a minimum and maximum value.  If the value is out of    */
/*           range, the return value is VI_TRUE, otherwise the return value  */
/*           is VI_FALSE.                                                    */
/*===========================================================================*/
ViBoolean gmmsp_invalidViReal64Range (ViReal64 val, ViReal64 min, ViReal64 max)
{
    return (ViBoolean)((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Check String                                                    */
/* Purpose:  This function checks a string value to see if it is             */
/*           of a defined length and initialized.                            */
/*===========================================================================*/
ViBoolean gmmsp_invalidViString (ViString val, ViUInt32 maxLength)
{
    if (val)
        return (ViBoolean)((strlen (val) > maxLength) ? VI_TRUE : VI_FALSE);

    return VI_TRUE;
}

/*===========================================================================*/
/* Function: Initialize Clean Up                                             */
/* Purpose:  This function is used only by the gmmsp_init function.  When   */
/*           an error is detected this function is called to close the       */
/*           open resource manager and instrument object sessions and to     */
/*           set the instrSession that is returned from gmmsp_init to       */
/*           VI_NULL.                                                        */
/*===========================================================================*/
ViStatus gmmsp_initCleanUp (ViSession openRMSession,
                            ViPSession openInstrSession,
                            ViStatus currentStatus)
{
    gmmsp_instrRange instrPtr;
    
    if (viGetAttribute (*openInstrSession, VI_ATTR_USER_DATA, &instrPtr) >= 0)
        if (instrPtr != NULL) 
            free (instrPtr);

    viClose (*openInstrSession);
    viClose (openRMSession);
    *openInstrSession = VI_NULL;
    
    return currentStatus;
}

/*****************************************************************************/
/*----------- INSERT INSTRUMENT-DEPENDENT UTILITY ROUTINES HERE -------------*/
/*****************************************************************************/

/*===========================================================================*/
/* Function: Default Instrument Setup                                        */
/* Purpose:  This function sends a default setup to the instrument.  This    */
/*           function is called by the gmmsp_reset operation and by the     */
/*           gmmsp_init function if the reset option has not been           */
/*           selected.  This function is useful for configuring any          */
/*           instrument settings that are required by the rest of the        */
/*           instrument driver functions such as turning headers ON or OFF   */
/*           or using the long or short form for commands, queries, and data.*/                                    
/*===========================================================================*/
ViStatus gmmsp_defaultInstrSetup (ViSession instrSession)
{
    ViStatus	gmmsp_status = VI_SUCCESS;
    ViChar		rdBuffer[256],
    			*p2buffer;
    ViInt32 	i, tmpValue;
    gmmsp_instrRange instrPtr;
    
    /* Determine if the structure has been initialized for the current VISA  */
    /* Session and malloc if it has not.                                     */
	CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));
    
    if (instrPtr == NULL) 
        instrPtr = malloc (sizeof (struct gmmsp_statusDataRanges));

    instrPtr -> ioDelay = DEFAULT_IO_DELAY;
	Delay(instrPtr->ioDelay);
    CHECKERR (viPrintf (instrSession, "CO?"));
	Delay(instrPtr->ioDelay);
	CHECKERR (viScanf (instrSession, "%256[^\n]", rdBuffer));
    p2buffer = strchr(rdBuffer, ' ') +1;
	
	for (i = 0; i < NUMBER_OF_CHANNELS; i++)
		{
		sscanf(p2buffer, "%2ld", &tmpValue);
		instrPtr -> voltageLimits[i] = (ViReal64)tmpValue;
		p2buffer += 2;
		sscanf(p2buffer, "%2ld", &tmpValue);
  		instrPtr -> currentLimits[i] = ((ViReal64)tmpValue)/10.0;
		p2buffer += 3;
		}

    instrPtr -> errorChecking = VI_TRUE;
    strcpy (instrPtr -> instrDriverRevision, GMMSP_REVISION);
    
    CHECKERR (viSetAttribute (instrSession, VI_ATTR_USER_DATA, (ViUInt32)instrPtr));

    return gmmsp_status;
}

/*=========================================================================*/
/* Function: Check Status                                                  */
/* Purpose:  This function reads the system status byte.                   */
/*=========================================================================*/
ViStatus gmmsp_checkStatus (ViSession instrSession)
{
    ViStatus	gmmsp_status = VI_SUCCESS;
    ViInt32		esrByte = 0;
    gmmsp_instrRange instrPtr;

    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));

    if (instrPtr -> errorChecking)
    	{ /* Is Error Checking Enabled? */
        /*--------------- Read the Standard Event Register ------------------*/
		Delay(instrPtr->ioDelay);
        CHECKERR (viPrintf (instrSession, "*ESR?"));
		Delay(instrPtr->ioDelay);
        CHECKERR (viScanf (instrSession, "%ld", &esrByte));
        /*---------------- Check if any error or message bit was asserted ---*/
        gmmsp_status = (esrByte & IEEE_ERROR_BITS) ? GMMSP_ERROR_INSTRUMENT_ERROR : VI_SUCCESS;
    	}

    return gmmsp_status;
}
/*=========================================================================*/
/* Function: Read Channel Value                                            */
/* Purpose:  This function reads value of specific channel                 */
/*=========================================================================*/
ViStatus gmmsp_readChannelValue (ViSession instrSession, 
								 ViString command, 
								 ViInt32 channel, 
								 ViPReal64 value)
{
    ViStatus	gmmsp_status = VI_SUCCESS;
	ViChar		tmpBuffer[BUFFER_SIZE],
				rdBuffer[BUFFER_SIZE],
    			*p2buffer;
  	gmmsp_instrRange instrPtr;    			

    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr))
  	Delay (instrPtr->ioDelay);
  	CHECKERR (viPrintf (instrSession, command));
  	Delay (instrPtr->ioDelay);
  	CHECKERR (viScanf (instrSession, "%256[^\n]", rdBuffer));
  	
	sprintf(tmpBuffer, "%c%ld,", (channel==1) ? ' ':',', channel);

    if ((p2buffer = strstr(rdBuffer, tmpBuffer)) != VI_NULL)
    	{
    	p2buffer += 3;
    	sscanf(p2buffer, "%Lf", value);
    	}
    else
    	return VI_ERROR_INSTR_INTERPRETING_RESPONSE;
    	
  	return gmmsp_status;
}

/*=========================================================================*/
/* Function: Read Channel State                                            */
/* Purpose:  This function reads state of specific channel                 */
/*=========================================================================*/
ViStatus gmmsp_readChannelState (ViSession instrSession, 
								 ViString command, 
								 ViInt32 channel, 
								 ViPBoolean state)
{
    ViStatus	gmmsp_status = VI_SUCCESS;
	ViChar		tmpBuffer[BUFFER_SIZE],
    			rdBuffer[256],
    			*p2buffer,
    			mode[256];
  	gmmsp_instrRange instrPtr;    			
    			
    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr))
  	Delay (instrPtr->ioDelay);
  	CHECKERR (viPrintf (instrSession, command));
  	Delay (instrPtr->ioDelay);
  	CHECKERR (viScanf (instrSession, "%256[^\n]", rdBuffer));	
    sprintf(tmpBuffer, "%ld,", channel);
	
    if ((p2buffer = strstr(rdBuffer, tmpBuffer)) != VI_NULL)
    	{
    	p2buffer += 2;
    	sscanf(p2buffer, "%256[^,]", mode);
    	}
    else
    	return VI_ERROR_INSTR_INTERPRETING_RESPONSE;
    	
    *state = (ViBoolean) (strstr( mode, "ON") ? VI_TRUE : VI_FALSE);
	
  	return gmmsp_status;
}

/*=========================================================================*/
/* Function: Write Group		                                           */
/* Purpose:  This function writes group                                    */
/*=========================================================================*/
ViStatus gmmsp_writeGroup (ViSession instrSession, 
						   ViInt32 group,	
						   ViString command)
{
	ViStatus	gmmsp_status = VI_SUCCESS;
    ViChar 		tmpBuffer[BUFFER_SIZE];
    ViInt32		length, i;
  	gmmsp_instrRange instrPtr;    			
    	
  	CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr));

	tmpBuffer[0] = '\0';
	for (i=0; i < NUMBER_OF_CHANNELS; i++)
		{
		if (group & (1<<i))
			sprintf(tmpBuffer,"%s%ld,",tmpBuffer,(i+1));
		}
	
	if ((length = strlen(tmpBuffer)) == 0)
		sprintf(tmpBuffer,"0");
	else
		tmpBuffer[length-1] = '\0';
		
  	Delay (instrPtr->ioDelay);
	CHECKERR (viPrintf (instrSession, "%s %s", command, tmpBuffer));
    
	return gmmsp_status;
}	

/*=========================================================================*/
/* Function: Read Group		                                               */
/* Purpose:  This function reads group                                     */
/*=========================================================================*/
ViStatus gmmsp_readGroup (ViSession instrSession, 
						  ViPInt32 group,	
						  ViString command)
{
	ViStatus	gmmsp_status = VI_SUCCESS;
    ViInt32		value;
    ViInt32		i;
    ViChar		rdBuffer[256],
    			*p2buffer;
  	gmmsp_instrRange instrPtr;    			

    *group = 0;
    			
    CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr))
    Delay (instrPtr->ioDelay);
  	CHECKERR (viPrintf (instrSession, command));
  	Delay (instrPtr->ioDelay);
  	CHECKERR (viScanf (instrSession, "%256[^\n]", rdBuffer));
  	
    if ((p2buffer = strstr(rdBuffer, " ")) != VI_NULL)
    	p2buffer += 1;
    else
    	return VI_ERROR_INSTR_INTERPRETING_RESPONSE;
    
	for (i = 0; i < NUMBER_OF_CHANNELS; i++)
		{
    	sscanf(p2buffer, "%ld", &value);
    	if (value)
	    	*group |= (1 << (value-1));
	    p2buffer += 2;	
		}
		
	return gmmsp_status;
}

/*=========================================================================*/
/* Function: Set I/O Delay Time                                            */
/* Purpose:  This function set a delay between each two instrument I/O     */
/*           operations.                                                   */
/*=========================================================================*/
ViStatus _VI_FUNC gmmsp_utilSetIODelay (ViSession instrSession,
                						ViReal64 IODelayTime) 
{
  	ViStatus gmmsp_status = VI_SUCCESS;
  	gmmsp_instrRange instrPtr;    
  	
	if (IODelayTime < 0.0)
    	 return VI_ERROR_PARAMETER2;  	
    
  	CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr))

  	instrPtr->ioDelay = IODelayTime;
    
  	return gmmsp_status;
}

/*=========================================================================*/
/* Function: Get I/O Delay Time                                            */
/* Purpose:  This function returns a delay between each two instrument I/O */
/*           operations.                                                   */
/*=========================================================================*/
ViStatus _VI_FUNC gmmsp_utilGetIODelay (ViSession instrSession,
						                ViPReal64 IODelayTime) 
{
	ViStatus gmmsp_status = VI_SUCCESS;
  	gmmsp_instrRange instrPtr;    

  	if (!IODelayTime)
    	 return VI_ERROR_PARAMETER2;  	

  	CHECKERR (viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr))

   	*IODelayTime = instrPtr->ioDelay;
    
  	return gmmsp_status;
}

/*=========================================================================*/
/* Function: Lock Instrument                                               */
/* Purpose:  This function locks/unlocks the instrument.                   */
/*=========================================================================*/
ViStatus _VI_FUNC gmmsp_utilLockInstr (ViSession instrSession, 
										ViBoolean instrument)
{
ViStatus gmmsp_status = VI_SUCCESS;
 
  if (gmmsp_invalidViBooleanRange (instrument))  
       return VI_ERROR_PARAMETER2;
  
  if (instrument)
	gmmsp_status = viGpibControlREN (instrSession, VI_GPIB_REN_ASSERT_LLO);   //lock 
  else
  	gmmsp_status = viGpibControlREN (instrSession, VI_GPIB_REN_DEASSERT_GTL);  //unlock

  return gmmsp_status;
}

/*****************************************************************************/
/*=== END INSTRUMENT DRIVER SOURCE CODE =====================================*/
/*****************************************************************************/
